import express from "express";
import path from "path";
import { fileURLToPath } from "url";
import fs from "fs";
import crypto from "crypto";
import dotenv from "dotenv";
import engine from "ejs-mate";
import { createShipmentSmartRule, getShipmentLabelsPdf } from "./src/sendy.js";
import { getConfig, updateStoredSettings, maskSecret } from "./src/config.js";

dotenv.config();

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();

app.engine("ejs", engine);
app.set("view engine", "ejs");
app.set("views", path.join(__dirname, "views"));

app.use(express.urlencoded({ extended: true }));
app.use(express.json());
app.use(express.static(path.join(__dirname, "public"), { etag: true, maxAge: "1h" }));

// Basic hardening headers (no extra deps)
app.use((req, res, next) => {
  res.setHeader("X-Content-Type-Options", "nosniff");
  res.setHeader("X-Frame-Options", "DENY");
  res.setHeader("Referrer-Policy", "no-referrer");
  // Note: Keep CSP off for now (EJS inline styles/scripts may be used).
  next();
});

const DATA_DIR = path.join(__dirname, "data");
const TX_FILE = path.join(DATA_DIR, "transactions.json");
const LABEL_DIR = path.join(DATA_DIR, "labels");

function toInt(value, fallback) {
  const n = Number.parseInt(String(value ?? ""), 10);
  return Number.isFinite(n) ? n : fallback;
}

// Ensure local data folders/files exist
if (!fs.existsSync(DATA_DIR)) fs.mkdirSync(DATA_DIR, { recursive: true });
if (!fs.existsSync(LABEL_DIR)) fs.mkdirSync(LABEL_DIR, { recursive: true });
if (!fs.existsSync(TX_FILE)) fs.writeFileSync(TX_FILE, "[]", "utf8");

const BRAND = {
  name: "PaySVP",
  primary: "#0b1220",
  accent: "#22c55e",
  brand: "#0b1220",
  logo: "/logo.svg"
};


function ensureDataFiles() {
  if (!fs.existsSync(DATA_DIR)) fs.mkdirSync(DATA_DIR, { recursive: true });
  if (!fs.existsSync(LABEL_DIR)) fs.mkdirSync(LABEL_DIR, { recursive: true });
  if (!fs.existsSync(TX_FILE)) {
    writeJson(TX_FILE, []);
  }
}

ensureDataFiles();

function readJson(file, fallback) {
  try { return JSON.parse(fs.readFileSync(file, "utf-8")); }
  catch { return fallback; }
}
function ensureDirForFile(file) {
  const dir = path.dirname(file);
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
}

function writeJson(file, data) {
  ensureDirForFile(file);
  const tmp = file + ".tmp";
  fs.writeFileSync(tmp, JSON.stringify(data, null, 2), "utf-8");
  fs.renameSync(tmp, file);
}

function isCacheExpired(filePath, ttlHours) {
  const ttl = Number(ttlHours);
  if (!Number.isFinite(ttl) || ttl <= 0) return false; // 0 = cache forever
  try {
    const st = fs.statSync(filePath);
    const ageMs = Date.now() - st.mtimeMs;
    return ageMs > ttl * 60 * 60 * 1000;
  } catch {
    return false;
  }
}


function slugify(input) {
  return String(input || "")
    .toLowerCase()
    .trim()
    .replace(/[^a-z0-9]+/g, "-")
    .replace(/^-+|-+$/g, "")
    .slice(0, 64) || "product";
}

function signLink(url) {
  const cfg = getConfig();
  const secret = cfg.PAYS_VP_SIGNING_SECRET || "change-me";
  const sig = crypto.createHmac("sha256", secret).update(url).digest("hex");
  const u = new URL(url);
  u.searchParams.set("sig", sig);
  return u.toString();
}

function buildPlainLink({ product, price, customer }) {
  const cfg = getConfig();
  const base = String(cfg.PAYS_VP_BASE_URL || process.env.PAYS_VP_BASE_URL || "https://paysvp.com").replace(/\/+$/,"");
  const p = slugify(product);
  const pr = String(price || "").replace(/[^\d.,]/g, "").replace(",", ".");
  const cust = customer ? encodeURIComponent(customer.trim()) : "customer";
  return `${base}/${p}/${pr}/${cust}`;
}

app.get("/", (req, res) => res.redirect("/admin/dashboard"));

app.get("/admin", (req, res) => res.redirect("/admin/dashboard"));

app.get("/admin/index", (req, res) => {
  const rows = readJson(TX_FILE, []);
  const since = Date.now() - (24 * 60 * 60 * 1000);
  const rows24 = rows.filter(r => {
    const t = Date.parse(r.paid_at || r.created_at || r.updated_at || "");
    return Number.isFinite(t) && t >= since;
  }).length;
  res.render("admin/index", { brand: BRAND, rows24 });
});

app.get("/admin/dashboard", (req, res) => {
  const rows = readJson(TX_FILE, []);
  res.render("admin/dashboard", { brand: BRAND, rows });
});

app.get("/admin/new", (req, res) => {
  res.render("admin/new", { brand: BRAND, form: {}, out: null });
});

app.post("/admin/new", (req, res) => {
  const { shop, price, product, signed } = req.body;
  const plain = buildPlainLink({ product, price, customer: shop });
  const out = { plain, signed: signed ? signLink(plain) : null };

  res.render("admin/new", { brand: BRAND, form: req.body, out });
});

app.get("/admin/sign", (req, res) => {
  res.render("admin/sign", { brand: BRAND, form: {}, out: null });
});

app.post("/admin/sign", (req, res) => {
  const { shop, price, product } = req.body;
  const plain = buildPlainLink({ product, price, customer: shop });
  const out = { plain, signed: signLink(plain) };
  res.render("admin/sign", { brand: BRAND, form: req.body, out });
});

app.get("/admin/pi/:pi", (req, res) => {
  const rows = readJson(TX_FILE, []);
  const pi = rows.find(r => r.pi === req.params.pi);
  if (!pi) return res.status(404).send("Not found");
  res.render("admin/pi", { brand: BRAND, pi, flash: null });
});


/**
 * Admin settings page (stores secrets locally in data/settings.json).
 * Leave a field blank to keep existing value.
 */
app.get("/admin/settings", (req, res) => {
  const cfg = getConfig();
  res.render("admin/settings", {
    brand: BRAND,
    active: "settings",
    cfg: {
      PAYS_VP_BASE_URL: cfg.PAYS_VP_BASE_URL || "",
      PAYS_VP_SIGNING_SECRET: maskSecret(cfg.PAYS_VP_SIGNING_SECRET),
      SENDY_BASE_URL: cfg.SENDY_BASE_URL || "https://app.sendy.nl",
      SENDY_TOKEN: maskSecret(cfg.SENDY_TOKEN),
      SENDY_SHOP_ID: cfg.SENDY_SHOP_ID ? maskSecret(cfg.SENDY_SHOP_ID) : "",
      STRIPE_SECRET_KEY: maskSecret(cfg.STRIPE_SECRET_KEY),
      STRIPE_WEBHOOK_SECRET: maskSecret(cfg.STRIPE_WEBHOOK_SECRET),

      SENDY_LABEL_CACHE_TTL_HOURS: String(cfg.SENDY_LABEL_CACHE_TTL_HOURS ?? "24"),
      SENDY_LABEL_CACHE_MAX_FILES: String(cfg.SENDY_LABEL_CACHE_MAX_FILES ?? "300")
    },
    flash: null
  });
});

app.post("/admin/settings", (req, res) => {
  try {
    const patch = {
      PAYS_VP_BASE_URL: req.body.PAYS_VP_BASE_URL,
      PAYS_VP_SIGNING_SECRET: req.body.PAYS_VP_SIGNING_SECRET,
      SENDY_BASE_URL: req.body.SENDY_BASE_URL,
      SENDY_TOKEN: req.body.SENDY_TOKEN,
      SENDY_SHOP_ID: req.body.SENDY_SHOP_ID,
      STRIPE_SECRET_KEY: req.body.STRIPE_SECRET_KEY,
      STRIPE_WEBHOOK_SECRET: req.body.STRIPE_WEBHOOK_SECRET,

      SENDY_LABEL_CACHE_TTL_HOURS: req.body.SENDY_LABEL_CACHE_TTL_HOURS,
      SENDY_LABEL_CACHE_MAX_FILES: req.body.SENDY_LABEL_CACHE_MAX_FILES
    };
    updateStoredSettings(patch);
    const cfg = getConfig();
    res.render("admin/settings", {
      brand: BRAND,
      active: "settings",
      cfg: {
        PAYS_VP_BASE_URL: cfg.PAYS_VP_BASE_URL || "",
        PAYS_VP_SIGNING_SECRET: maskSecret(cfg.PAYS_VP_SIGNING_SECRET),
        SENDY_BASE_URL: cfg.SENDY_BASE_URL || "https://app.sendy.nl",
        SENDY_TOKEN: maskSecret(cfg.SENDY_TOKEN),
        SENDY_SHOP_ID: cfg.SENDY_SHOP_ID ? maskSecret(cfg.SENDY_SHOP_ID) : "",
        STRIPE_SECRET_KEY: maskSecret(cfg.STRIPE_SECRET_KEY),
        STRIPE_WEBHOOK_SECRET: maskSecret(cfg.STRIPE_WEBHOOK_SECRET),
        SENDY_LABEL_CACHE_TTL_HOURS: String(cfg.SENDY_LABEL_CACHE_TTL_HOURS ?? "24"),
        SENDY_LABEL_CACHE_MAX_FILES: String(cfg.SENDY_LABEL_CACHE_MAX_FILES ?? "300")
      },
      flash: { type: "ok", message: "Instellingen opgeslagen." }
    });
  } catch (e) {
    res.status(500).render("admin/settings", {
      brand: BRAND,
      active: "settings",
      cfg: {
        PAYS_VP_BASE_URL: "",
        PAYS_VP_SIGNING_SECRET: "",
        SENDY_BASE_URL: "https://app.sendy.nl",
        SENDY_TOKEN: "",
        SENDY_SHOP_ID: "",
        STRIPE_SECRET_KEY: "",
        STRIPE_WEBHOOK_SECRET: "",
        SENDY_LABEL_CACHE_TTL_HOURS: "24",
        SENDY_LABEL_CACHE_MAX_FILES: "300"
      },
      flash: { type: "error", message: e.message || "Kon instellingen niet opslaan." }
    });
  }
});

/**
 * Cache tools
 */
function listCachedLabelFiles() {
  if (!fs.existsSync(LABEL_DIR)) return [];
  return fs.readdirSync(LABEL_DIR)
    .filter(f => f.toLowerCase().endsWith(".pdf"))
    .map(f => {
      const full = path.join(LABEL_DIR, f);
      const st = fs.statSync(full);
      return { name: f, path: full, mtimeMs: st.mtimeMs, size: st.size };
    })
    .sort((a, b) => b.mtimeMs - a.mtimeMs);
}

function enforceCacheLimit(maxFiles) {
  if (!maxFiles || maxFiles <= 0) return;
  const files = listCachedLabelFiles();
  if (files.length <= maxFiles) return;
  const toDelete = files.slice(maxFiles); // already sorted newest-first
  for (const f of toDelete) {
    try { fs.unlinkSync(f.path); } catch {}
  }
}

app.get("/admin/cache", (req, res) => {
  const cfg = getConfig();
  const files = listCachedLabelFiles();
  const totalBytes = files.reduce((a, f) => a + f.size, 0);
  res.render("admin/cache", {
    brand: BRAND,
    active: "cache",
    stats: {
      files: files.length,
      totalBytes,
      ttlHours: toInt(cfg.SENDY_LABEL_CACHE_TTL_HOURS, 24),
      maxFiles: toInt(cfg.SENDY_LABEL_CACHE_MAX_FILES, 300)
    },
    recent: files.slice(0, 25)
  });
});

app.post("/admin/cache/clear", (req, res) => {
  const files = listCachedLabelFiles();
  for (const f of files) {
    try { fs.unlinkSync(f.path); } catch {}
  }
  res.redirect("/admin/cache");
});


/**
 * Create Sendy shipment for a PI (smart-rule).
 * Requires SENDY_TOKEN and SENDY_SHOP_ID in env.
 */
app.post("/admin/pi/:pi/sendy", async (req, res) => {
  const rows = readJson(TX_FILE, []);
  const idx = rows.findIndex(r => r.pi === req.params.pi);
  if (idx === -1) return res.status(404).send("Not found");

  const pi = rows[idx];
  const cfg = getConfig();

  try {
    const shipment = await createShipmentSmartRule({
      token: cfg.SENDY_TOKEN,
      baseUrl: cfg.SENDY_BASE_URL || "https://app.sendy.nl",
      payload: {
        shop_id: cfg.SENDY_SHOP_ID,
        country: (pi.country || process.env.DEFAULT_COUNTRY || "NL"),
        company_name: pi.company_name || undefined,
        contact: (pi.name || "").slice(0, 35) || undefined,
        street: (pi.street || "").slice(0, 35),
        number: String(pi.number || pi.nummer || "").slice(0, 10) || "",
        addition: (pi.addition || pi.toevoeging || "").slice(0, 10) || undefined,
        postal_code: (pi.postal_code || pi.postcode || "").slice(0, 12) || "",
        city: (pi.city || "").slice(0, 35) || "",
        phone: (pi.phone || "").slice(0, 30) || undefined,
        email: pi.email || undefined,
        reference: (`${pi.shop || "PaySVP"} ${pi.product || ""} ${pi.pi}`).slice(0, 35),
        weight: pi.weight_kg || 0.2,
        amount: pi.amount_packages || 1
      }
    });

    // Store shipment UUID + portal URL if present
    pi.sendy = {
      uuid: shipment?.data?.uuid || shipment?.uuid,
      url: shipment?.data?.url || shipment?.url,
      created_at: new Date().toISOString()
    };
    rows[idx] = pi;
    writeJson(TX_FILE, rows);

    res.redirect(`/admin/pi/${encodeURIComponent(pi.pi)}`);
  } catch (e) {
    res.status(500);
    res.render("admin/pi", { brand: BRAND, pi, flash: { type: "error", message: e.message || "Sendy error" } });
  }
});


/**
 * Download Sendy shipment label PDF for a PI.
 * Requires an existing pi.sendy.uuid.
 */
app.get("/admin/pi/:pi/sendy/labels.pdf", async (req, res) => {
  const rows = readJson(TX_FILE, []);
  const idx = rows.findIndex(r => r.pi === req.params.pi);
  if (idx === -1) return res.status(404).send("Not found");
  const pi = rows[idx];
  if (!pi.sendy?.uuid) return res.status(400).send("Geen Sendy shipment voor deze PI.");

  const cfg = getConfig();

  // Cache path based on shipment uuid
  const uuid = String(pi.sendy.uuid);
  const cacheFile = path.join(LABEL_DIR, `${uuid}.pdf`);
  const ttlHours = toInt(cfg.SENDY_LABEL_CACHE_TTL_HOURS, 24);
  const maxFiles = toInt(cfg.SENDY_LABEL_CACHE_MAX_FILES, 300);

  try {
    // Serve cached label if exists & still within TTL
    if (fs.existsSync(cacheFile)) {
      if (ttlHours > 0) {
        const st = fs.statSync(cacheFile);
        const ageMs = Date.now() - st.mtimeMs;
        if (ageMs > ttlHours * 60 * 60 * 1000) {
          try { fs.unlinkSync(cacheFile); } catch {}
        }
      }
    }

    if (fs.existsSync(cacheFile)) {
      const safePi = String(pi.pi || "shipment").replace(/[^a-zA-Z0-9_-]+/g, "_");
      res.setHeader("Content-Type", "application/pdf");
      res.setHeader("Content-Disposition", `attachment; filename=label-${safePi}.pdf`);
      return fs.createReadStream(cacheFile).pipe(res);
    }

    const { pdfBuffer } = await getShipmentLabelsPdf({
      token: cfg.SENDY_TOKEN,
      baseUrl: cfg.SENDY_BASE_URL || "https://app.sendy.nl",
      uuid
    });

    // Write cache
    if (!fs.existsSync(LABEL_DIR)) fs.mkdirSync(LABEL_DIR, { recursive: true });
    fs.writeFileSync(cacheFile, pdfBuffer);

    // Enforce max-files limit after writing
    enforceCacheLimit(maxFiles);

    // Store cache metadata on PI (optional)
    pi.sendy.label_cached_at = new Date().toISOString();
    pi.sendy.label_path = `data/labels/${uuid}.pdf`;
    rows[idx] = pi;
    writeJson(TX_FILE, rows);

    const safePi = String(pi.pi || "shipment").replace(/[^a-zA-Z0-9_-]+/g, "_");
    res.setHeader("Content-Type", "application/pdf");
    res.setHeader("Content-Disposition", `attachment; filename=label-${safePi}.pdf`);
    res.send(pdfBuffer);
  } catch (e) {
    res.status(500);
    res.send(e.message || "Sendy label error");
  }
});




app.get("/admin/export.csv", (req, res) => {
  const rows = readJson(TX_FILE, []);
  const header = ["pi","status","amount","currency","shop","product","name","email","street","number","postal_code","city","country","paid_at"].join(",");
  const lines = rows.map(r => ([
    r.pi, r.status, r.amount, r.currency, r.shop, r.product, r.name, r.email,
    r.street, (r.number||r.nummer), (r.postal_code||r.postcode), r.city, r.country,
    r.paid_at
  ].map(v => `"${String(v??"").replace(/"/g,'""')}"`).join(",")));
  res.setHeader("Content-Type", "text/csv; charset=utf-8");
  res.setHeader("Content-Disposition", "attachment; filename=export.csv");
  res.send([header, ...lines].join("\n"));
});

app.get("/admin/export.json", (req, res) => {
  const rows = readJson(TX_FILE, []);
  res.setHeader("Content-Type", "application/json; charset=utf-8");
  res.setHeader("Content-Disposition", "attachment; filename=export.json");
  res.send(JSON.stringify(rows, null, 2));
});




// Fallback error handler (keeps errors readable in browser)
app.use((err, req, res, next) => {
  const msg = err?.message || "Unexpected server error";
  if (res.headersSent) return next(err);
  res.status(500).send(msg);
});
const port = Number(process.env.PORT || 3000);
app.listen(port, () => {
  console.log(`PaySVP admin running on http://localhost:${port}`);
});