import fs from "fs";
import path from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Stored settings file (local, not committed)
const SETTINGS_FILE = path.join(__dirname, "..", "data", "settings.json");

function safeReadJson(file, fallback = {}) {
  try {
    if (!fs.existsSync(file)) return fallback;
    const raw = fs.readFileSync(file, "utf8");
    const parsed = JSON.parse(raw);
    return (parsed && typeof parsed === "object") ? parsed : fallback;
  } catch {
    return fallback;
  }
}

function safeWriteJsonAtomic(file, obj) {
  const dir = path.dirname(file);
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
  const tmp = file + ".tmp";
  fs.writeFileSync(tmp, JSON.stringify(obj, null, 2), "utf8");
  fs.renameSync(tmp, file);
}

/**
 * Returns merged config: env defaults overridden by stored settings.json.
 */
export function getConfig() {
  const stored = safeReadJson(SETTINGS_FILE, {});
  return {
    PAYS_VP_BASE_URL: stored.PAYS_VP_BASE_URL ?? process.env.PAYS_VP_BASE_URL,
    PAYS_VP_SIGNING_SECRET: stored.PAYS_VP_SIGNING_SECRET ?? process.env.PAYS_VP_SIGNING_SECRET,

    SENDY_BASE_URL: stored.SENDY_BASE_URL ?? process.env.SENDY_BASE_URL ?? "https://app.sendy.nl",
    SENDY_TOKEN: stored.SENDY_TOKEN ?? process.env.SENDY_TOKEN,
    SENDY_SHOP_ID: stored.SENDY_SHOP_ID ?? process.env.SENDY_SHOP_ID,

    STRIPE_SECRET_KEY: stored.STRIPE_SECRET_KEY ?? process.env.STRIPE_SECRET_KEY,
    STRIPE_WEBHOOK_SECRET: stored.STRIPE_WEBHOOK_SECRET ?? process.env.STRIPE_WEBHOOK_SECRET,

    // Sendy label cache settings
    // - TTL hours: cached labels older than this will be refetched
    // - Set to 0 to disable TTL (cache forever)
    SENDY_LABEL_CACHE_TTL_HOURS:
      (stored.SENDY_LABEL_CACHE_TTL_HOURS ?? process.env.SENDY_LABEL_CACHE_TTL_HOURS ?? "24"),

    // Hard limit: keep at most N cached PDFs (oldest deleted first). 0 = unlimited.
    SENDY_LABEL_CACHE_MAX_FILES:
      (stored.SENDY_LABEL_CACHE_MAX_FILES ?? process.env.SENDY_LABEL_CACHE_MAX_FILES ?? "300")
  };
}

export function readStoredSettings() {
  return safeReadJson(SETTINGS_FILE, {});
}

/**
 * Partial update. Empty strings are ignored (keeps existing).
 */
export function updateStoredSettings(patch) {
  const current = safeReadJson(SETTINGS_FILE, {});
  const next = { ...current };

  for (const [k, v] of Object.entries(patch || {})) {
    if (v === undefined || v === null) continue;
    const s = String(v).trim();
    if (!s) continue; // keep existing
    next[k] = s;
  }

  safeWriteJsonAtomic(SETTINGS_FILE, next);
  return next;
}

export function maskSecret(value) {
  if (!value) return "";
  const s = String(value);
  if (s.length <= 8) return "••••••••";
  return s.slice(0, 4) + "••••••••" + s.slice(-4);
}
