/**
 * Sendy.nl API helper
 *
 * Auth: Bearer Personal Access Token (SENDY_TOKEN) or OAuth access token.
 * Base URL: https://app.sendy.nl  (or your configured SENDY_BASE_URL)
 *
 * Relevant endpoints:
 *  - POST /api/shipments/smart-rule
 *  - GET  /api/shipments/{uuid}/labels   -> { labels: <base64 PDF> }
 */
function cleanBaseUrl(baseUrl) {
  return String(baseUrl || "https://app.sendy.nl").replace(/\/+$/,"");
}

async function requestJson({ url, method="GET", token, body, timeoutMs = 15000 }) {
  if (!token) throw new Error("SENDY_TOKEN ontbreekt (Personal Access Token).");

  const controller = new AbortController();
  const t = setTimeout(() => controller.abort(), timeoutMs);

  let resp;
  try {
    resp = await fetch(url, {
      method,
      signal: controller.signal,
      headers: {
        "Authorization": `Bearer ${token}`,
        "Content-Type": "application/json",
        "Accept": "application/json"
      },
      body: body ? JSON.stringify(body) : undefined
    });
  } catch (e) {
    if (e?.name === "AbortError") throw new Error("Sendy API timeout.");
    throw e;
  } finally {
    clearTimeout(t);
  }

  const text = await resp.text();
  let data = null;
  try { data = text ? JSON.parse(text) : null; }
  catch { data = { raw: text }; }

  if (!resp.ok) {
    const msg =
      (data && (data.message || data.error || data.error_description)) ||
      `HTTP ${resp.status}`;
    throw new Error(`Sendy API fout: ${msg}`);
  }
  return data;
}

export async function createShipmentSmartRule({ baseUrl, token, payload }) {
  if (!payload?.shop_id) throw new Error("SENDY_SHOP_ID ontbreekt (shop_id).");
  if (!payload?.country) throw new Error("country ontbreekt.");
  const url = `${cleanBaseUrl(baseUrl)}/api/shipments/smart-rule`;
  return requestJson({ url, method: "POST", token, body: payload });
}

/**
 * Fetch shipment labels as a PDF Buffer.
 * Returns: { pdfBuffer, raw } where raw is the JSON response.
 */
export async function getShipmentLabelsPdf({ baseUrl, token, uuid }) {
  if (!uuid) throw new Error("Shipment UUID ontbreekt.");
  const url = `${cleanBaseUrl(baseUrl)}/api/shipments/${encodeURIComponent(uuid)}/labels`;
  const raw = await requestJson({ url, method: "GET", token });

  const b64 = raw?.labels || raw?.data?.labels;
  if (!b64 || typeof b64 !== "string") {
    throw new Error("Geen labels ontvangen van Sendy.");
  }

  let pdfBuffer;
  try {
    pdfBuffer = Buffer.from(b64, "base64");
  } catch {
    throw new Error("Labels zijn geen geldige base64.");
  }

  // Basic sanity check: PDF header
  if (pdfBuffer.length < 5 || pdfBuffer.slice(0, 4).toString("utf8") !== "%PDF") {
    // Sometimes Sendy may return a PDF without header? Unlikely, but keep it helpful.
    // Don't hard-fail: still return the buffer.
  }

  return { pdfBuffer, raw };
}
